// NAVBulkOperationChainInfo
NAVBulkOperationChainInfo.prototype.arrayFilters = function (arrayFilters) {
    if (isUndefined(arrayFilters))
        nav_throwError("bulk.find.arrayFilters() requires filter array");
    if (!isObject(arrayFilters))
        nav_throwError("arrayFilters should be object");

    return this.forwardToCustomFunction("arrayFilters", arrayFilters);
};

NAVBulkOperationChainInfo.prototype.collation = function (collationDoc) {
    if (isUndefined(collationDoc))
        nav_throwError("bulk.find.collation() requires collation document");
    if (!isObject(collationDoc))
        nav_throwError("collation document should be object");

    return this.forwardToCustomFunction("collation", collationDoc);
};

NAVBulkOperationChainInfo.prototype.execute = function (writeConcern) {
    if (!isUndefined(writeConcern)) {
        var w = writeConcern.w;
        if (!isUndefined(w)) {
            if (!isNumber(w) && !isString(w)) {
                nav_throwError("write concern require valid w option");
            }
        }

        var j = writeConcern.j;
        if (!isUndefined(j)) {
            if (!isBoolean(j)) 
               nav_throwError("write concern require j to be true or false");
        }

        var wtimeout = writeConcern.wtimeout;
        if (!isUndefined(wtimeout)) {
            if (!isNumber(wtimeout))
                nav_throwError("write concern require wtimeout to be number");
        }
        for (var key in writeConcern) {
            if (key != "w" && key != "j" && key != "wtimeout") {
                nav_throwError("unrecognized write concern field: " + key);
            }
        }
    }
    var bulkResult = this.forwardToCustomFunction("execute", writeConcern);
    if (isNumber(bulkResult.nRemoved)) {
        var bulkWriteResult = new BulkWriteResult(bulkResult, JSON.parse(this.tojson()).nBatches == 1 ? this.getOperations().batches[0].batchType : null, writeConcern)
        if (bulkWriteResult.hasErrors())
            throw bulkWriteResult.toError();

        return bulkWriteResult;
    }
    return bulkResult;
};

NAVBulkOperationChainInfo.prototype.find = function (selector) {
    if (isUndefined(selector))
        nav_throwError("bulk.find() requires selector");
    if (!isObject(selector))
        nav_throwError("selector should be object");

    return this.forwardToCustomFunction("find", selector);
};

NAVBulkOperationChainInfo.prototype.getOperations = function () {
    return this.forwardToCustomFunction("getOperations");
};

NAVBulkOperationChainInfo.prototype.insert = function (insertDoc) {
    if (isUndefined(insertDoc))
        nav_throwError("bulk.insert() requires insert document");
    if (!isObject(insertDoc))
        nav_throwError("insert document should be object");

    if (isUndefined(insertDoc._id) && !Array.isArray(insertDoc)) {
        var objOld = insertDoc;
        insertDoc = { _id: new ObjectId() };
        Object.extend(insertDoc, objOld);
    }

    return this.forwardToCustomFunction("insert", insertDoc);
};

NAVBulkOperationChainInfo.prototype.remove = function () {
    return this.forwardToCustomFunction("remove");
};
NAVBulkOperationChainInfo.prototype.delete = NAVBulkOperationChainInfo.prototype.remove;

NAVBulkOperationChainInfo.prototype.removeOne = function () {
    return this.forwardToCustomFunction("removeOne");
};
NAVBulkOperationChainInfo.prototype.deleteOne = NAVBulkOperationChainInfo.prototype.removeOne;

NAVBulkOperationChainInfo.prototype.replaceOne = function (replacementDoc) {
    if (isUndefined(replacementDoc))
        nav_throwError("bulk.find.replaceOne() requires replacement document");
    if (Array.isArray(replacementDoc))
        nav_throwError('bulk.find.replaceOne.replaceOne does not support pipeline-style update"');
    if (!isObject(replacementDoc))
        nav_throwError("replacement document should be object");

    return this.forwardToCustomFunction("replaceOne", replacementDoc);
};

NAVBulkOperationChainInfo.prototype.setLetParams = function (userLet) {
    return this.forwardToCustomFunction("letParams", userLet);
};

NAVBulkOperationChainInfo.prototype.tojson = function () {
    return this.forwardToCustomFunction("tojson");
};
NAVBulkOperationChainInfo.prototype.toString = NAVBulkOperationChainInfo.prototype.tojson;
NAVBulkOperationChainInfo.prototype.toJSON = NAVBulkOperationChainInfo.prototype.tojson;

NAVBulkOperationChainInfo.prototype.update = function (updateDoc) {
    if (isUndefined(updateDoc))
        nav_throwError("bulk.find.update() requires update document");
    if (!isObject(updateDoc) && !Array.isArray(updateDoc))
        nav_throwError("update document should be object or pipeline");

    return this.forwardToCustomFunction("update", updateDoc);
};

NAVBulkOperationChainInfo.prototype.updateOne = function (updateDoc) {
    if (isUndefined(updateDoc))
        nav_throwError("bulk.find.updateOne() requires update document");
    if (!isObject(updateDoc) && !Array.isArray(updateDoc))
        nav_throwError("update document should be object or pipeline");

    return this.forwardToCustomFunction("updateOne", updateDoc);
};

NAVBulkOperationChainInfo.prototype.upsert = function () {
    return this.forwardToCustomFunction("upsert");
};

NAVBulkOperationChainInfo.prototype.hint = function (hint) {
    return this.forwardToCustomFunction("hint", hint);
};

// BulkWriteResult
function _defineReadOnlyProperty(self, name, value) {
    Object.defineProperty(self, name, {
        enumerable: true,
        get: function () {
            return value;
        }
    });
};

var BulkWriteResult = function (bulkResult, singleBatchType, writeConcern) {

    if (!(this instanceof BulkWriteResult) && !(this instanceof BulkWriteError))
        return new BulkWriteResult(bulkResult, singleBatchType, writeConcern);

    _defineReadOnlyProperty(this, "ok", bulkResult.ok);
    _defineReadOnlyProperty(this, "nInserted", bulkResult.nInserted);
    _defineReadOnlyProperty(this, "nUpserted", bulkResult.nUpserted);
    _defineReadOnlyProperty(this, "nMatched", bulkResult.nMatched);
    _defineReadOnlyProperty(this, "nModified", bulkResult.nModified? bulkResult.nModified : 0);
    _defineReadOnlyProperty(this, "nRemoved", bulkResult.nRemoved);

    this.getUpsertedIds = function () {
        return bulkResult.upserted? bulkResult.upserted : [];
    };

    this.getUpsertedIdAt = function (index) {
        return bulkResult.upserted? bulkResult.upserted[index] : null;
    };

    this.getRawResponse = function () {
        return bulkResult;
    };

    this.hasWriteErrors = function () {
        return bulkResult.writeErrors ? bulkResult.writeErrors.length > 0 : false;
    };

    this.getWriteErrorCount = function () {
        return this.hasWriteErrors() ? bulkResult.writeErrors.length : 0;
    };

    this.getWriteErrorAt = function (index) {
        if (index < bulkResult.writeErrors.length)
            return new WriteError(bulkResult.writeErrors[index]);
        return null;
    };

    this.getWriteErrors = function () {
        return bulkResult.writeErrors;
    };

    this.hasWriteConcernError = function () {
        return bulkResult.writeConcernErrors? bulkResult.writeConcernErrors.length > 0 : false;
    };
    
    this.getWriteConcernErrorCount = function () {
        return this.hasWriteConcernError()? bulkResult.writeConcernErrors.length : 0;
    };

    this.getWriteConcernErrorAt = function (index) {
        if (this.hasWriteConcernError() && index < bulkResult.writeConcernErrors.length)
            return new WriteConcernError(bulkResult.writeConcernErrors[index]);
        return null;
    };

    this.getWriteConcernError = function () {
        if (this.hasWriteConcernError()) {
            if (bulkResult.writeConcernErrors.length == 1) {
                return new WriteConcernError(bulkResult.writeConcernErrors[0]);
            } else {
                var errmsg = "";
                for (var i = 0; i < bulkResult.writeConcernErrors.length; i++) {
                    var err = bulkResult.writeConcernErrors[i];
                    errmsg = errmsg + err.errmsg;
                    if (i != bulkResult.writeConcernErrors.length - 1) {
                        errmsg = errmsg + " and ";
                    }
                }

                var WRITE_CONCERN_FAILED = 64;
                return new WriteConcernError({ errmsg: errmsg, code: WRITE_CONCERN_FAILED });
            }
        }
        return null;
    };

    this.tojson = function (indent, nolint) {
        return tojson(bulkResult, indent, nolint);
    };

    this.toString = function () {
        if (writeConcern && writeConcern.w == 0) {
            return "BulkWriteResult(" + tojson({}) + ")";
        }
        return "BulkWriteResult(" + this.tojson() + ")";
    };

    this.shellPrint = function () {
        return this.toString();
    };

    this.hasErrors = function () {
        return this.hasWriteErrors() || this.hasWriteConcernError();
    };

    this.toError = function () {
        if (this.hasErrors()) {
            var message = undefined;
            return new BulkWriteError(bulkResult, singleBatchType, writeConcern, message);
        } else {
            nav_throwError("batch was successful, cannot create BulkWriteError");
        }
    };

    this.nav_toError = function () {
        if (this.hasErrors()) {
            var tmpBulkWriteError = this.toError();
            var tmpStr = '';
            if (tmpBulkWriteError.name)
                tmpStr = tmpBulkWriteError.name + ":\n" + tmpBulkWriteError.toString();
            else if (tmpBulkWriteError.message)
                tmpStr = tmpBulkWriteError.message + "\n" + tmpBulkWriteError.toString();
            return tmpStr;
        }
        return null;
    }

    this.toSingleResult = function () {
        if (singleBatchType == null)
            nav_throwError("Cannot output single WriteResult from multiple batch result");
        return new WriteResult(bulkResult, singleBatchType, writeConcern);
    };
};

var WriteResult = function (bulkResult, singleBatchType, writeConcern) {

    if (!(this instanceof WriteResult))
        return new WriteResult(bulkResult, singleBatchType, writeConcern);

    _defineReadOnlyProperty(this, "ok", bulkResult.ok);
    _defineReadOnlyProperty(this, "nInserted", bulkResult.nInserted);
    _defineReadOnlyProperty(this, "nUpserted", bulkResult.nUpserted);
    _defineReadOnlyProperty(this, "nMatched", bulkResult.nMatched);
    _defineReadOnlyProperty(this, "nModified", bulkResult.nModified);
    _defineReadOnlyProperty(this, "nRemoved", bulkResult.nRemoved);
    if (bulkResult.upserted.length > 0)
        _defineReadOnlyProperty(this, "_id", bulkResult.upserted[bulkResult.upserted.length - 1]._id);

    this.getUpsertedId = function () {
        if (bulkResult.upserted == null || bulkResult.upserted.length == 0)
            return null;

        return bulkResult.upserted[bulkResult.upserted.length - 1];
    };

    this.getRawResponse = function () {
        return bulkResult;
    };

    this.hasWriteError = function () {
        return bulkResult.writeErrors ? bulkResult.writeErrors.length > 0 : false;
    };

    this.getWriteErrorCount = function () {
        return this.hasWriteError() ? bulkResult.writeErrors.length : 0;
    };

    this.getWriteError = function () {
        if (bulkResult.writeErrors.length == 0)
            return null;
        else
            return bulkResult.writeErrors[bulkResult.writeErrors.length - 1];
    };

    this.hasWriteConcernError = function () {
        return bulkResult.writeConcernErrors ? bulkResult.writeConcernErrors.length > 0 : false;
    };

    this.getWriteConcernErrorCount = function () {
        return this.hasWriteConcernError() ? bulkResult.writeConcernErrors.length : 0;
    };

    this.getWriteConcernErrorAt = function (index) {
        if (this.hasWriteConcernError() && index < bulkResult.writeConcernErrors.length)
            return bulkResult.writeConcernErrors[index];
        return null;
    };

    this.getWriteConcernError = function () {
        if (this.hasWriteConcernError())
            return bulkResult.writeConcernErrors[0];
        return null;
    };

    this.tojson = function (indent, nolint) {
        var result = {};

        if (singleBatchType == 1) {
            result.nInserted = this.nInserted;
        }

        if (singleBatchType == 2) {
            result.nMatched = this.nMatched;
            result.nUpserted = this.nUpserted;

            if (this.nModified != undefined)
                result.nModified = this.nModified;

            if (Array.isArray(bulkResult.upserted) && bulkResult.upserted.length == 1) {
                result._id = bulkResult.upserted[0]._id;
            }
        }

        if (singleBatchType == 3) {
            result.nRemoved = bulkResult.nRemoved;
        }

        if (this.hasWriteError()) {
            result.writeError = {};
            result.writeError.code = this.getWriteError().code;
            result.writeError.errmsg = this.getWriteError().errmsg;
        }

        if (this.hasWriteConcernError() != null) {
            result.writeConcernError = [];
            for (var i = 0; i < this.getWriteConcernErrorCount(); i++) {
                var wcError = {};
                var wc = this.getWriteConcernErrorAt(i);
                wcError.code = wc.code;
                wcError.errmsg = wc.errmsg;
                result.writeConcernError.push(wcError);
            }
        }

        return tojson(result, indent, nolint);
    };

    this.toString = function () {
        if (writeConcern && writeConcern.w == 0) {
            return "WriteResult(" + tojson({}) + ")";
        }
        return "WriteResult(" + this.tojson() + ")";
    };

    this.shellPrint = function () {
        return this.toString();
    };

    this.hasErrors = function () {
        return this.hasWriteError() || this.hasWriteConcernError();
    };

    this.nav_toError = function () {
        if (this.hasErrors()) {
            var tmpStr = '';
            if (this.hasWriteError() && this.hasWriteConcernError()) {
                var message = undefined;
                var tmpBulkWriteError = new BulkWriteError(bulkResult, singleBatchType, writeConcern, message);
                if (tmpBulkWriteError.name)
                    tmpStr = tmpBulkWriteError.name + ":\n" + tmpBulkWriteError.toString();
                else if (tmpBulkWriteError.message)
                    tmpStr = tmpBulkWriteError.message + "\n" + tmpBulkWriteError.toString();
            } else if (this.hasWriteError()) {
                var err = this.getWriteError();
                tmpStr = err.errmsg ? "index " + err.index + ": " + err.code + " - " + err.errmsg : 'unknown write error';
            } else if (this.hasWriteConcernError()) {
                var wcErr = this.getWriteConcernError();
                tmpStr = wcErr.errmsg ? wcErr.code + " - " + wcErr.errmsg : 'unknown write concern error';
            }
            return tmpStr;
        }
        return null;
    }
};

var BulkWriteError = function (bulkResult, singleBatchType, writeConcern, message) {

    if (!(this instanceof BulkWriteError))
        return new BulkWriteError(bulkResult, singleBatchType, writeConcern, message);

    BulkWriteResult.apply(this, arguments);

    this.name = undefined;
    this.message = undefined;
    if (this.getWriteErrorCount() + this.getWriteConcernErrorCount() > 1) {
        if (this.getWriteErrorCount() > 0)
            this.name = 'BulkWriteError';
        else
            this.name = 'WriteConcernError';
    } else if (this.getWriteErrorCount() == 1) {
        var err = bulkResult.writeErrors[0];
        this.message = 'Index ' + err.index + ': ' + err.code + ' - ' + err.errmsg;
    } else if (this.getWriteConcernErrorCount() == 1) {
        var err = bulkResult.writeConcernErrors[0];
        this.message = err.code + ' - ' + err.errmsg;
    }

    delete this.toError;

    this.toString = function () {
        var outputMsg = '';
        if (isUndefined(this.message)) {
            for (var i = 0; i < this.getWriteErrorCount() ; ++i) {
                var err = bulkResult.writeErrors[i];
                outputMsg += 'Index ' + err.index + ': ' + err.code + ' - ' + err.errmsg + '\n';
            }
            if (this.hasWriteConcernError()) {
                outputMsg += 'WriteConcernError: \n';
                for (var i = 0; i < this.getWriteConcernErrorCount() ; i++) {
                    var wcError = this.getWriteConcernErrorAt(i);
                    outputMsg += wcError.code + ' - ' + wcError.errmsg + '\n';
                }
            }
        }

        outputMsg += 'nInserted : ' + bulkResult.nInserted + ', ' +
            'nUpserted : ' + bulkResult.nUpserted + ', ' +
            'nMatched : ' + bulkResult.nMatched + ', ' +
            'nModified : ' + bulkResult.nModified + ', ' +
            'nRemoved : ' + bulkResult.nRemoved
        
        if (!isUndefined(bulkResult.upserted)) {
            outputMsg += '\n upserted : ' + tojson(bulkResult.upserted);
        }

        return outputMsg;
    };
    this.stack = this.toString() + "\n" + (new Error().stack);

    this.toResult = function () {
        return new BulkWriteResult(bulkResult, singleBatchType, writeConcern);
    };
};

BulkWriteError.prototype = Object.create(Error.prototype);
BulkWriteError.prototype.constructor = BulkWriteError;

var WriteCommandError = function (commandError) {
    if (!(this instanceof WriteCommandError))
        return new WriteCommandError(commandError);

    // Define properties
    _defineReadOnlyProperty(this, "code", commandError.code);
    _defineReadOnlyProperty(this, "errmsg", commandError.errmsg);
    if (commandError.hasOwnProperty("errorLabels")) {
        _defineReadOnlyProperty(this, "errorLabels", commandError.errorLabels);
    }

    this.name = 'WriteCommandError';
    this.message = this.errmsg;

    this.tojson = function (indent, nolint) {
        return tojson(commandError, indent, nolint);
    };

    this.toString = function () {
        return "WriteCommandError(" + this.tojson() + ")";
    };
    this.stack = this.toString() + "\n" + (new Error().stack);

    this.shellPrint = function () {
        return this.toString();
    };
};

WriteCommandError.prototype = Object.create(Error.prototype);
WriteCommandError.prototype.constructor = WriteCommandError;

var WriteError = function (err) {
    if (!(this instanceof WriteError))
        return new WriteError(err);

    this.name = undefined;
    this.message = err.errmsg ? "index " + err.index + ": " + err.code + " - " + err.errmsg : 'unknown write error';

    _defineReadOnlyProperty(this, "code", err.code);
    _defineReadOnlyProperty(this, "index", err.index);
    _defineReadOnlyProperty(this, "errmsg", err.errmsg);
    if (err.hasOwnProperty("errInfo"))
        _defineReadOnlyProperty(this, "errInfo", err.errInfo);

    this.getOperation = function () {
        return err.op;
    };

    this.tojson = function (indent, nolint) {
        return tojson(err, indent, nolint);
    };

    this.toString = function () {
        return "";
    };
    this.stack = this.toString() + "\n" + (new Error().stack);

    this.shellPrint = function () {
        return this.toString();
    };
};

WriteError.prototype = Object.create(Error.prototype);
WriteError.prototype.constructor = WriteError;

var WriteConcernError = function (err) {
    if (!(this instanceof WriteConcernError))
        return new WriteConcernError(err);

    this.name = undefined;
    this.message = err.errmsg ? err.code + " - " + err.errmsg : 'unknown write concern error';

    _defineReadOnlyProperty(this, "code", err.code);
    _defineReadOnlyProperty(this, "errInfo", err.errInfo);
    _defineReadOnlyProperty(this, "errmsg", err.errmsg);

    this.tojson = function (indent, nolint) {
        return tojson(err, indent, nolint);
    };

    this.toString = function () {
        return "";
    };
    this.stack = this.toString() + "\n" + (new Error().stack);

    this.shellPrint = function () {
        return this.toString();
    };
};

WriteConcernError.prototype = Object.create(Error.prototype);
WriteConcernError.prototype.constructor = WriteConcernError;
